const express = require("express");
const router = express.Router();

module.exports = (products) => {

    // "View" products page
    router.get("/", async (req, res) => {
        const allProducts = await products.find().toArray();
        res.render("index", { products: allProducts });
    });

    // "Add" product page
    router.get("/add", (req, res) => res.render("add"));

    // Handle an add 
    router.post("/add", async (req, res) => {
        // Get the attributes from the request body
        const { name, category, price, stock, ageRange } = req.body;

        // Find the highest _id of all products
        const lastProduct = await products.findOne({}, { sort: { _id: -1 } });

        // Make sure to do an increment on the _id each time
        const nextId = lastProduct ? lastProduct._id + 1 : 1;

        // Now add the product
        await products.insertOne({
            _id: nextId,
            name,
            category,
            price: parseFloat(price),
            stock: parseInt(stock),
            ageRange
        });

        // Go back to the products view page
        res.redirect("/products");
    });

    // "Update" product page
    router.get("/update", async (req, res) => {
        const allProducts = await products.find().toArray();
        res.render("update", { products: allProducts });
    });

    // Handle an update
    router.post("/update", async (req, res) => {
        const { id, name, category, price, stock, ageRange } = req.body;

        // Update the product now
        await products.updateOne(
            { _id: parseInt(id) },
            { $set: { name, category, price: parseFloat(price), stock: parseInt(stock), ageRange } }
        );

        // Go back to the products view page
        res.redirect("/products");
    });

    // "Delete" product page
    router.get("/delete", async (req, res) => {
        const allProducts = await products.find().toArray();
        res.render("delete", { products: allProducts });
    });

    // Handle a delete
    router.post("/delete", async (req, res) => {
        const { id } = req.body;
        await products.deleteOne({ _id: parseInt(id) });

        // Go back to the products view page
        res.redirect("/products");
    });

    // "Search" products page
    router.get("/search", (req, res) => res.render("search", { products: [] }));

    // Handle a search
    router.post("/search", async (req, res) => {
        const { field, value } = req.body;
        const query = {};
        if (field && value) query[field] = value;
        const results = await products.find(query).toArray();
        res.render("search", { products: results });
    });

    return router;
};
