// This code will connect to a database add some toys to a products collection 
// and then perform some queries/updates and then quit.  It is similar to our
// MongoDB example test, except that we will use a schema to perform validation
// checks and insert one product at a time so that we can see the individual errors
// and ensure that all good products are added. Also, we removed the fixed IDs from
// our toy data, since IDs are usually supplied by the database in practice.

// Get the database connect/disconnect functions 
const { connectToDatabase, disconnectFromDatabase } = require("./mongooseDB");
const Product = require("./ToyProductModel");
const toys = require("./toysData");  // the toys to insert

// We made a main() function because we are required to use "async"
async function main() {
    try {
        const db = await connectToDatabase();

        // Clear the collection first each time we run, since this is just a test
        await Product.deleteMany({});

        // Insert the toys in the above-defined array
        let count = 0;
        for (const t of toys) { 
            try {
                let result = await Product.create(t);
                console.log("Added " + result.name);
                count++;
            }
            catch(err) {
                console.log("*** Error: Did not add " + t.name);
                for (const e in err.errors) {
                    console.log("   Path:", err.errors[e].path);       
                    console.log("   Kind:", err.errors[e].kind);     
                    console.log("   Value:", err.errors[e].value);      
                    console.log("   Message:", err.errors[e].message); 
                }
            }
        }
        console.log(count + " products were added.");

        // Display all added products
        let prods = await Product.find();
        console.log("Here are the products that were added:");
        for (const p of prods) { 
            console.log(p);
        }

        // Display all products with a price under $30
        let cheap = await Product.find({price:{$lt:30}}, {name: 1, price:1});
        console.log("Here are the products under $30:");
        for (const p of cheap) { 
            console.log(p);
        }

        // Update the price of the Barbie Dreamhouse
        result = await Product.updateOne({name:"Barbie Dreamhouse"}, {$set: {price: 179.99}});
        if (result.acknowledged && result.modifiedCount > 0)
            console.log("Price has been updated for Barbie Dreamhouse");
        else
            console.log("Price has NOT been updated for Barbie Dreamhouse");

        // Verify the price of the Barbie Dreamhouse
        let product = await Product.findOne({name:"Barbie Dreamhouse"});
        console.log("Price is now at $" + product.price);

        // Replace the NERF Blaster Elite with a new version of the product
        result = await Product.replaceOne({name: "NERF Blaster Elite"}, 
            {name: "RC Monster Truck", category: "Remote Control", price: 59.99, stock: 40, ageRange: "6+"});
        if (result.acknowledged)
            console.log("NERF Blaster Elite has been replaced by RC Monster Truck");
        else
            console.log("NERF Blaster Elite has NOT been replaced");
    } catch (err) {
        console.error("Error ... something went wrong:", err);
        for (const e in err.errors) {
            console.log(err.errors[e].kind);    
            console.log(err.errors[e].message); 
            console.log(err.errors[e].path);    
            console.log(err.errors[e].value);   
        }

    } finally {
        await disconnectFromDatabase();
    }
}

main();